import os
import json
import torch
import argparse
from tqdm import tqdm
from datasets import load_dataset
from transformers import AutoTokenizer, AutoConfig, AutoModelForCausalLM
import re
import sys
import random
import numpy as np


torch.manual_seed(42)
torch.cuda.manual_seed(42)
torch.cuda.manual_seed_all(42)
np.random.seed(42)
random.seed(42)
torch.backends.cudnn.deterministic = True
torch.backends.cudnn.benchmark = False

sys.path.append("/fs/nexus-scratch/hjae/ShadowKV")
from models.llama import LlamaForCausalLM  
from models.qwen import Qwen2ForCausalLM  

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--model_path", type=str, default="deepseek-ai/DeepSeek-R1-Distill-Qwen-7B")
    parser.add_argument("--token_budget", type=int, default=1024)
    parser.add_argument("--compression_enabled", action="store_true")
    parser.add_argument("--compression_threshold", type=int, default=128)
    parser.add_argument("--compression_ratio", type=float, default=0.5)
    parser.add_argument("--window_size", type=int, default=512) 
    parser.add_argument("--max_samples", type=int, default=100)
    parser.add_argument("--output_dir", type=str, default="results")
    return parser.parse_args()

def load_model_and_tokenizer(args):
    print(f"Loading model from {args.model_path}...")
    tokenizer = AutoTokenizer.from_pretrained(
        args.model_path,
        use_fast=False
    )
    config = AutoConfig.from_pretrained(args.model_path)

    if "qwen" in args.model_path.lower() or "deepseek" in args.model_path.lower():
        print("Using Qwen2ForCausalLM with ShadowKV support")
        model = Qwen2ForCausalLM.from_pretrained(
            args.model_path,
            config=config,
            device_map="auto",
            torch_dtype=torch.float16
        )
    else:
        print("Using LlamaForCausalLM with ShadowKV support")
        model = LlamaForCausalLM.from_pretrained(
            args.model_path,
            config=config,
            device_map="auto",
            torch_dtype=torch.float16
        )
    
    if hasattr(model, "shadowkv_init"):
        print("Initializing ShadowKV...")
        model.shadowkv_init(
            window_size=args.window_size,
            max_tokens=args.token_budget,
            compress_ratio=args.compression_ratio if args.compression_enabled else 1.0,
            compress_threshold=args.compression_threshold,
        )
        print(f"ShadowKV initialized with compression_enabled={args.compression_enabled}")
    else:
        print("Warning: ShadowKV is not available for this model")
        if args.compression_enabled:
            print("Compression will be disabled as ShadowKV is not supported")
    
    return model, tokenizer

def format_prompt(premises, conclusion):
    return f"""Read the following premises and determine whether the conclusion logically follows. Answer with one of: True, False, or Uncertain.\n\nPremises: {premises}\n\nConclusion: {conclusion}\n\nAnswer:"""

def extract_answer(response):
    match = re.search(r"(True|False|Uncertain)", response, re.IGNORECASE)
    if match:
        return match.group(1).capitalize()
    return None

def is_correct(pred, gold):
    if pred is None:
        return False
    return pred.lower() == gold.lower()

def load_partial_results(output_file):
    if os.path.exists(output_file):
        with open(output_file, "r") as f:
            data = json.load(f)
            return data.get("results", []), data.get("correct", 0), data.get("total", 0)
    return [], 0, 0

def evaluate_folio(model, tokenizer, dataset, args, output_file):
    results, correct, total = load_partial_results(output_file)
    start_idx = total
    n_total = min(len(dataset), args.max_samples) if args.max_samples else len(dataset)
    batch_size = 100
    save_every = 50

    while start_idx < n_total:
        end_idx = min(start_idx + batch_size, n_total)
        subset = dataset.select(range(start_idx, end_idx))
        
        for sample in tqdm(subset, initial=start_idx, total=n_total):
            premises = sample["premises"]
            conclusion = sample["conclusion"]
            gold = sample["label"]

            prompt = format_prompt(premises, conclusion)
            inputs = tokenizer(prompt, return_tensors="pt").to(model.device)

            with torch.no_grad():
                outputs = model.generate(
                    inputs.input_ids,
                    max_new_tokens=64,  
                    temperature=0.7,
                    top_p=0.9,
                    use_cache=True
                )

            response = tokenizer.decode(outputs[0], skip_special_tokens=True)
            predicted_answer = extract_answer(response)

            correct_flag = is_correct(predicted_answer, gold)
            if correct_flag:
                correct += 1
            total += 1

            results.append({
                "premises": premises,
                "conclusion": conclusion,
                "gold_label": gold,
                "predicted_answer": predicted_answer,
                "is_correct": correct_flag,
                "response": response
            })

            if (total % save_every == 0) or (total == n_total):
                with open(output_file, "w") as f:
                    json.dump({
                        "args": vars(args),
                        "accuracy": correct / total if total > 0 else 0,
                        "results": results,
                        "correct": correct,
                        "total": total
                    }, f, indent=2)
                print(f"[Checkpoint] Saved {total} results to {output_file}")
        
        start_idx = end_idx
    
    return results, correct / total if total > 0 else 0

def main():
    args = parse_args()
    os.makedirs(args.output_dir, exist_ok=True)

    model, tokenizer = load_model_and_tokenizer(args)

    print("Loading FOLIO dataset...")
    dataset = load_dataset("yale-nlp/FOLIO")
    test_set = dataset["validation"] if "validation" in dataset else dataset["test"] if "test" in dataset else dataset["train"]

    if args.max_samples is not None:
        test_set = test_set.shuffle(seed=42).select(range(min(args.max_samples, len(test_set))))
        print(f"Using {len(test_set)} samples for evaluation (randomly selected with seed=42)")

    print("Starting evaluation...")
    output_file = os.path.join(args.output_dir, "folio_results_shadowkv_deepseekw.json")
    results, accuracy = evaluate_folio(model, tokenizer, test_set, args, output_file)

    print(f"\nFinal accuracy: {accuracy:.2%}")
    print(f"Results saved to {output_file}")

if __name__ == "__main__":
    main()
